package gov.cms.grouper.snf.lego;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.List;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import gov.cms.grouper.snf.lego.SnfComparator.AcceptNull;
import gov.cms.grouper.snf.lego.SnfComparator.NullAs;

public class SnfComparatorTest {

  public static final int fail = Integer.MIN_VALUE;
  public static final int pass = Integer.MAX_VALUE;

  @Test
  public void testLessThan() {
    Compare<Integer> com = AcceptNull.COMPUTE_NULL_AS_HIGH.ofCompare();
    Integer thisValue = null;
    Integer isLessThanThis = null;
    boolean expected = false;
    boolean actual = com.lt(thisValue, isLessThanThis);
    Assertions.assertEquals(expected, actual);

    thisValue = null;
    isLessThanThis = null;
    expected = true;
    actual = com.lte(thisValue, isLessThanThis);
    Assertions.assertEquals(expected, actual);

    thisValue = null;
    isLessThanThis = 1;
    expected = false;
    actual = com.lt(thisValue, isLessThanThis);
    Assertions.assertEquals(expected, actual);

    thisValue = null;
    isLessThanThis = 1;
    expected = false;
    actual = com.lte(thisValue, isLessThanThis);
    Assertions.assertEquals(expected, actual);

    thisValue = 1;
    isLessThanThis = 1;
    expected = false;
    actual = com.lt(thisValue, isLessThanThis);
    Assertions.assertEquals(expected, actual);

    thisValue = 1;
    isLessThanThis = 1;
    expected = true;
    actual = com.lte(thisValue, isLessThanThis);
    Assertions.assertEquals(expected, actual);

    thisValue = 2;
    isLessThanThis = 1;
    expected = false;
    actual = com.lt(thisValue, isLessThanThis);
    Assertions.assertEquals(expected, actual);

    thisValue = 2;
    isLessThanThis = 1;
    expected = false;
    actual = com.lte(thisValue, isLessThanThis);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testNullAs() {
    NullAs<Integer> as = NullAs.of(Integer.MIN_VALUE);
    Integer value = null;
    Integer expected = as.getNullValue();
    Integer actual = as.ck(value);
    Assertions.assertEquals(expected, actual);

    value = 1;
    expected = value;
    actual = as.ck(value);
    Assertions.assertEquals(expected, actual);


  }

  @Test
  public void testEq() {
    Compare<Integer> com = AcceptNull.COMPUTE_NULL_AS_HIGH.ofCompare();

    Integer other = null;
    Integer thisValue = null;
    boolean expected = true;
    boolean actual = com.eq(thisValue, other);
    Assertions.assertEquals(expected, actual);

    other = null;
    thisValue = 1;
    expected = false;
    actual = com.eq(thisValue, other);
    Assertions.assertEquals(expected, actual);

    other = 1;
    thisValue = null;
    expected = false;
    actual = com.eq(thisValue, other);
    Assertions.assertEquals(expected, actual);

    other = 1;
    thisValue = 0;
    expected = false;
    actual = com.eq(thisValue, other);
    Assertions.assertEquals(expected, actual);

    other = 0;
    thisValue = 1;
    expected = false;
    actual = com.eq(thisValue, other);
    Assertions.assertEquals(expected, actual);

    com = AcceptNull.FALSE_ON_NULL.ofCompare();
    other = null;
    thisValue = null;
    expected = false;
    actual = com.eq(thisValue, other);
    Assertions.assertEquals(expected, actual);

    other = 1;
    thisValue = 1;
    expected = true;
    actual = com.eq(thisValue, other);
    Assertions.assertEquals(expected, actual);

    com = AcceptNull.EXECEPTION_ON_NULL.ofCompare();
    try {
      other = 1;
      thisValue = null;
      expected = false;
      actual = com.eq(thisValue, other);
      Assertions.fail();
    } catch (Exception ex) {

    }

    try {
      other = null;
      thisValue = 1;
      expected = false;
      actual = com.eq(thisValue, other);
      Assertions.fail();
    } catch (Exception ex) {

    }

    try {
      other = null;
      thisValue = null;
      expected = false;
      actual = com.eq(thisValue, other);
      Assertions.fail();
    } catch (Exception ex) {

    }

    other = 1;
    thisValue = 2;
    expected = false;
    actual = com.eq(thisValue, other);
    Assertions.assertEquals(expected, actual);

    other = 1;
    thisValue = 1;
    expected = true;
    actual = com.eq(thisValue, other);
    Assertions.assertEquals(expected, actual);



  }

  @Test
  public void testGreaterThan() {
    Compare<Integer> com = AcceptNull.COMPUTE_NULL_AS_HIGH.ofCompare();
    Integer isGreaterThan = null;
    Integer thisValue = null;
    boolean expected = false;
    boolean actual = com.gt(thisValue, isGreaterThan);
    Assertions.assertEquals(expected, actual);

    isGreaterThan = null;
    thisValue = null;
    expected = true;
    actual = com.gte(thisValue, isGreaterThan);
    Assertions.assertEquals(expected, actual);

    isGreaterThan = null;
    thisValue = 1;
    expected = false;
    actual = com.gt(thisValue, isGreaterThan);
    Assertions.assertEquals(expected, actual);

    isGreaterThan = null;
    thisValue = 1;
    expected = false;
    actual = com.gte(thisValue, isGreaterThan);
    Assertions.assertEquals(expected, actual);

    isGreaterThan = 1;
    thisValue = 1;
    expected = false;
    actual = com.gt(thisValue, isGreaterThan);
    Assertions.assertEquals(expected, actual);

    isGreaterThan = 1;
    thisValue = 1;
    expected = true;
    actual = com.gte(thisValue, isGreaterThan);
    Assertions.assertEquals(expected, actual);

    isGreaterThan = 2;
    thisValue = 1;
    expected = false;
    actual = com.gt(thisValue, isGreaterThan);
    Assertions.assertEquals(expected, actual);

    isGreaterThan = 2;
    thisValue = 1;
    expected = false;
    actual = com.gte(thisValue, isGreaterThan);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testNullHigh() {
    List<Integer> list = Arrays.asList(1, 9, 5, 2, 5, null, 7);
    Integer[] expecteds = new Integer[] {1, 2, 5, 5, 7, 9, null};
    list.sort(SnfComparator.NULL_HIGH);
    Integer actuals[] = list.toArray(new Integer[] {});
    Assertions.assertArrayEquals(expecteds, actuals);
    System.out.println(list.toString());

    list = Arrays.asList(1, null, 9, -34, 2, 5, null, 7, 345);
    expecteds = new Integer[] {-34, 1, 2, 5, 7, 9, 345, null, null};
    list.sort(SnfComparator.NULL_HIGH);
    actuals = list.toArray(new Integer[] {});
    Assertions.assertArrayEquals(expecteds, actuals);
    System.out.println(list.toString());

    list = Arrays.asList(1, null, 9, -34, 2, 5, null, 7, 345);
    expecteds = new Integer[] {null, null, 345, 9, 7, 5, 2, 1, -34};
    list.sort(SnfComparator.NULL_HIGH.reversed());
    actuals = list.toArray(new Integer[] {});
    Assertions.assertArrayEquals(expecteds, actuals);
    System.out.println(list.toString());

  }

  @Test
  public void testNullLow() {
    List<Integer> list = Arrays.asList(1, 9, 5, 2, 5, null, 7);
    Integer[] expecteds = new Integer[] {null, 1, 2, 5, 5, 7, 9};
    list.sort(SnfComparator.NULL_LOW);
    Integer actuals[] = list.toArray(new Integer[] {});
    Assertions.assertArrayEquals(expecteds, actuals);
    System.out.println(list.toString());

    list = Arrays.asList(1, null, 9, -34, 2, 5, null, 7, 345);
    expecteds = new Integer[] {null, null, -34, 1, 2, 5, 7, 9, 345};
    list.sort(SnfComparator.NULL_LOW);
    actuals = list.toArray(new Integer[] {});
    Assertions.assertArrayEquals(expecteds, actuals);
    System.out.println(list.toString());

    list = Arrays.asList(1, null, 9, -34, 2, 5, null, 7, 345);
    expecteds = new Integer[] {345, 9, 7, 5, 2, 1, -34, null, null};
    list.sort(SnfComparator.NULL_LOW.reversed());
    actuals = list.toArray(new Integer[] {});
    Assertions.assertArrayEquals(expecteds, actuals);
    System.out.println(list.toString());

  }

  @Test
  public void testBetween() {
    BetweenCompare<Integer> com =
        BetweenCompare.of(AcceptNull.COMPUTE_NULL_AS_HIGH, AcceptNull.COMPUTE_NULL_AS_HIGH);
    Integer from = 290;
    Integer to = null;
    Integer check = 500;
    boolean expected = true;
    boolean actual = com.between(from, check, to);
    Assertions.assertEquals(expected, actual);

    from = null;
    to = null;
    check = Integer.MAX_VALUE;
    expected = false;
    actual = com.between(from, check, to);
    Assertions.assertEquals(expected, actual);

    from = 290;
    to = 500;
    check = 290;
    expected = true;
    actual = com.between(from, check, to);
    Assertions.assertEquals(expected, actual);

    from = 290;
    to = 500;
    check = 500;
    expected = false;
    actual = com.between(from, check, to);
    Assertions.assertEquals(expected, actual);

    from = null;
    to = null;
    check = null;
    expected = false;
    actual = com.between(from, check, to);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testBetweenRandom() {
    BetweenCompare<Integer> com =
        BetweenCompare.of(AcceptNull.COMPUTE_NULL_AS_HIGH, AcceptNull.EXECEPTION_ON_NULL);

    Integer from = 290;
    Integer to = Integer.MAX_VALUE;
    Integer check = 500;
    boolean expected = true;
    boolean actual = com.between(from, check, to);
    Assertions.assertEquals(expected, actual);

    int result = 0;
    try {
      from = -100;
      to = 100;
      check = null;
      expected = false;
      com.between(from, check, to);
    } catch (Exception ex) {
      result = pass;
    }
    Assertions.assertEquals(result, pass);

    com = BetweenCompare.of(AcceptNull.FALSE_ON_NULL, AcceptNull.EXECEPTION_ON_NULL);
    result = 0;
    try {
      from = -100;
      to = 100;
      check = null;
      expected = false;
      actual = com.between(from, check, to);
      Assertions.fail();
    } catch (Exception ex) {
      result = pass;
    }
    Assertions.assertEquals(result, pass);

    from = -100;
    to = 100;
    check = Integer.MAX_VALUE;
    expected = false;
    actual = com.between(from, check, to);

    from = -100;
    to = 100;
    check = 50;
    expected = true;
    actual = com.between(from, check, to);


  }


  @Test
  public void testBetweenInclusive() {
    BetweenCompare<Integer> com =
        BetweenCompare.of(AcceptNull.COMPUTE_NULL_AS_HIGH, AcceptNull.COMPUTE_NULL_AS_HIGH);
    Integer from = 290;
    Integer to = null;
    Integer check = 500;
    boolean expected = true;
    boolean actual = com.betweenInclusive(from, check, to);
    Assertions.assertEquals(expected, actual);

    from = null;
    to = null;
    check = Integer.MAX_VALUE;
    expected = false;
    actual = com.betweenInclusive(from, check, to);
    Assertions.assertEquals(expected, actual);

    from = 290;
    to = 500;
    check = 290;
    expected = true;
    actual = com.betweenInclusive(from, check, to);
    Assertions.assertEquals(expected, actual);

    from = 290;
    to = 500;
    check = null;
    expected = false;
    actual = com.betweenInclusive(from, check, to);
    Assertions.assertEquals(expected, actual);


    from = 290;
    to = 500;
    check = 500;
    expected = true;
    actual = com.betweenInclusive(from, check, to);
    Assertions.assertEquals(expected, actual);

    from = null;
    to = null;
    check = null;
    expected = true;
    actual = com.betweenInclusive(from, check, to);
    Assertions.assertEquals(expected, actual);

  }

  @Test
  public void testSum() {
    BigDecimal expected = new BigDecimal("10");
    List<Integer> inputs = Arrays.asList(1, 2, 3, 4);
    BigDecimal actual = SnfComparator.sum(inputs);
    Assertions.assertEquals(expected, actual);
  }

  @Test
  public void testAvg() {
    BigDecimal expected = new BigDecimal("3.33");
    List<Integer> inputs = Arrays.asList(5, 2, 3);
    BigDecimal actual = SnfComparator.avg(inputs, RoundingMode.HALF_UP, 2);
    Assertions.assertEquals(expected, actual);
  }


}
